import { SafeAreaView, Alert, TextInput } from "react-native";
import React, {
  forwardRef,
  useImperativeHandle,
  useState,
} from "react";
import {
  addDoc,
  collection,
  doc,
  getDocs,
  query,
  updateDoc,
  where,
} from "firebase/firestore";
import { auth, db } from "../../../config/firebase";
import {
  DEFAULT_FLATLIST_SCROLLVIEW_BOTTOM_PADDING,
  INPUT_CLASS_BORDER_BOTTOM,
} from "../../styles/constants";
import moment from "moment";
import PageLoader from "../../components/Loaders/PageLoader";
import { COLORS } from "../../styles/colors";
import { cleanString } from "../../utils/allFunctions";
import sendAdminNotif from "../../utils/sendAdminNotif";
import { useNavigation } from "@react-navigation/native";
import { useTranslation } from "react-i18next";
import { KeyboardAwareScrollView } from "react-native-keyboard-aware-scroll-view";
import { SENDGRID_API_KEY, SENDGRID_FROM } from "@env";
import axios from "axios";

const SearchStep3 = forwardRef((props, ref) => {
  const { formData, userInfo } = props;
  const { t, i18n } = useTranslation();

  const [vehicleId, setVehicleId] = useState("");
  const [assurance, setAssurance] = useState("");

  const [isLoading, setIsLoading] = useState(false);

  const navigation = useNavigation();

  const combinedData = Object.values(formData).reduce((acc, current) => {
    return { ...acc, ...current };
  }, {});


  function enleverEspaces(chaine) {
    return chaine.replace(/\s/g, "");
  }

  async function sendPushNotifToSearcher() {
    const message = {
      to: userInfo.expoPushToken,
      sound: "default",
      title: "Vous avez un match ! ✅",
      body: "Votre signalement au sein de l'application Sur4plots a permis de retrouver votre véhicule... RDV dans l'onglet 'Démarches'",
    };

    await fetch("https://exp.host/--/api/v2/push/send", {
      method: "POST",
      headers: {
        Accept: "application/json",
        "Accept-encoding": "gzip, deflate",
        "Content-Type": "application/json",
      },
      body: JSON.stringify(message),
    });
  }

  const sendMailToSearcher = async () => {
    try {
      const response = await axios.post(
        "https://api.sendgrid.com/v3/mail/send",
        {
          personalizations: [
            {
              to: [
                {
                  email: `${auth?.currentUser?.email}`,
                },
              ],
              dynamic_template_data: {
                first_name: auth?.currentUser?.displayName,
              },
            },
          ],
          from: {
            email: `${SENDGRID_FROM}`,
          },
          template_id: enleverEspaces("d-d8a99978650a4a6e9149b3cd7fd445de"),
        },
        {
          headers: {
            authorization: `Bearer ${SENDGRID_API_KEY}`,
            "Content-Type": "application/json",
          },
        }
      );

      console.log("Email sent successfully:", response.data);
    } catch (error) {
      if (error.response) {
        console.log("Error response data:", error.response.data);
        console.log("Error response status:", error.response.status);
        console.log("Error response headers:", error.response.headers);
      } else if (error.request) {
        console.log("Error request:", error.request);
      } else {
        console.log("Error message:", error.message);
      }
      console.log("Error config:", error.config);
    }
  };

  async function sendNotifToSearcher() {
    const dataToAdd = {
      title: "Vous avez un match ! ✅",
      text: "Votre signalement au sein de l'application Sur4plots a permis de retrouver votre véhicule... RDV dans l'onglet 'Démarches'",
      userId: auth?.currentUser?.uid,
      isViewed: false,
      createdAt: moment().format(),
    };

    await addDoc(collection(db, "notifications"), dataToAdd);
  }

  useImperativeHandle(ref, () => ({
    async validate() {
      setIsLoading(true);
      if (!vehicleId) {
        Alert.alert(
          "Veuillez rentrer votre plaque d'immatriculation afin de continuer la démarche"
        );
        setIsLoading(false);
        return;
      }

      if (!assurance) {
        Alert.alert("Veuillez rentrer votre assurance");
        setIsLoading(false);
        return;
      }

      try {
        setIsLoading(true);

        const q = query(
          collection(db, "searchs"),
          where("plate", "==", cleanString(vehicleId.toUpperCase())),
          where("userId", "==", null),
          where("status", "==", "progress")
        );
        const querySnapshot = await getDocs(q);
        const dataArr = [];
        if (querySnapshot.size > 0) {
          querySnapshot.forEach((res) => {
            dataArr.push({ id: res.id, ...res.data() });
          });
          const findedSearchDemand = dataArr[0];

          await updateDoc(doc(db, "searchs", findedSearchDemand.id), {
            country: combinedData.selectedCountry,
            type: combinedData.cat,
            assurance: assurance,
            status: "ended",
            userId: auth?.currentUser.uid,
            finderPaymentStatus: "unpaid",
            createdAt: moment().format(),
          });

          Alert.alert(
            "Votre véhicule a été retrouvé",
            "RDV dans l'onglet Démarches pour pouvoir consulter les photos et la dernière position où il a été aperçu",
            [
              {
                text: "J'ai compris",
                onPress: () => navigation.goBack(),
              },
            ],
            {
              cancelable: true,
            }
          );
          sendPushNotifToSearcher();
          sendMailToSearcher();
          sendNotifToSearcher();
          sendAdminNotif(`Un chercheur a trouvé son véhicule: ${vehicleId}`);
        } else {
          await addDoc(collection(db, "searchs"), {
            country: combinedData.selectedCountry,
            assurance: assurance,
            plate: cleanString(vehicleId),
            type: combinedData.cat,
            userId: auth?.currentUser.uid,
            finderId: null,
            status: "progress",
            price: "",
            finderPrice: "",
            paymentStatus: "unpaid",
            finderPaymentStatus: "unpaid",
            matchAnimation: false,
            createdAt: moment().format(),
          });

          Alert.alert(
            "Votre recherche a bien été prise en compte",
            "Vous serez notifié sous peu dès qu'on aura du nouveau concernant votre véhicule",
            [
              {
                text: "OK",
                onPress: () => navigation.goBack(),
              },
            ],
            {
              cancelable: true,
            }
          );
          sendAdminNotif(`Nouvelle recherche de véhicule: ${vehicleId}`);
        }
      } catch (error) {
        Alert.alert("Une erreur est survenue", `${error}`);
        return;
      } finally {
        setIsLoading(false);
      }

      setIsLoading(false);
    },
  }));

  if (isLoading) {
    return <PageLoader />;
  }

  return (
    <SafeAreaView className="flex-1 bg-white">
      <KeyboardAwareScrollView
        className="px-4"
        keyboardDismissMode="interactive"
        keyboardShouldPersistTaps="handled"
        scrollEnabled
        behavior="padding"
        contentContainerStyle={{
          paddingBottom: DEFAULT_FLATLIST_SCROLLVIEW_BOTTOM_PADDING,
        }}
      >
        <TextInput
          className={INPUT_CLASS_BORDER_BOTTOM}
          style={{
            fontFamily: "Inter_400Regular",
            marginTop: 20,
            color: COLORS.black,
          }}
          placeholder={t("text.immatriculation")}
          placeholderTextColor={"rgba(87, 96, 111,1.0)"}
          value={vehicleId}
          onChangeText={(text) => setVehicleId(text.toUpperCase())}
        />

        <TextInput
          className={INPUT_CLASS_BORDER_BOTTOM}
          style={{
            fontFamily: "Inter_400Regular",
            marginTop: 10,
            color: COLORS.black,
          }}
          placeholder={t("text.assurance")}
          placeholderTextColor={"rgba(87, 96, 111,1.0)"}
          value={assurance}
          onChangeText={setAssurance}
        />
      </KeyboardAwareScrollView>
    </SafeAreaView>
  );
});
export default SearchStep3;
